######################################################
#   Data Generation
######################################################

# A unidimensional model with 1 factor, 6 variables, 7-point scales, 300 subjects
# Standardized loadings are .8, .8, .7, .7, .6, and .4.
# The generation of the 7-point scales involved categorizing continuous data,
# assuming different cutoffs.


#~~~~~~~~~~~~~~~~ generate continuous variables ~~~~~~~~~~~~~~~~~#

set.seed(654321)
n = 300

f =  rnorm(n, 6, 1)
e1 = rnorm(n, 0, 1)
e2 = rnorm(n, 0, 1)
e3 = rnorm(n, 0, 1)
e4 = rnorm(n, 0, 1)
e5 = rnorm(n, 0, 1)
e6 = rnorm(n, 0, 1)

v1 = .8*f +  sqrt(1-.8^2)*e1
v2 = .7*f +  sqrt(1-.7^2)*e2
v3 = .7*f +  sqrt(1-.7^2)*e3
v4 = .4*f +  sqrt(1-.4^2)*e4
v5 = .8*f +  sqrt(1-.8^2)*e5
v6 = .6*f +  sqrt(1-.6^2)*e6

#~~~~~~~~~ categorize continuous data into 7-point scales ~~~~~~~~~~~#


F.SEVEN.UNI = function(x){
  xnew = cut(x,
             breaks = c(quantile(x, probs = c(0, 0.142, 0.285, 0.428, 0.571, 0.714, 0.857, 1))),
             labels = c(1, 2, 3, 4, 5, 6, 7),
             include.lowest = TRUE)
}

F.SEVEN.BEL = function(x){
  xnew = cut(x,
             breaks = c(quantile(x, probs = c(0, 0.0161, 0.0991, 0.3325, 0.6606, 0.8981, 0.9833, 1))),
             labels = c(1, 2, 3, 4, 5, 6, 7),
             include.lowest = TRUE)
}

F.SEVEN.RAN1 = function(x){
  xnew = cut(x,
             breaks = c(quantile(x, probs = c(0, 0.3, 0.6, 0.9, 0.97, 0.98, 0.99, 1))),
             labels = c(1, 2, 3, 4, 5, 6, 7),
             include.lowest = TRUE)
}

F.SEVEN.RAN2 = function(x){
  xnew = cut(x,
             breaks = c(quantile(x, probs = c(0, 0.01, 0.02, 0.03, 0.10, 0.40, 0.70, 1))),
             labels = c(1, 2, 3, 4, 5, 6, 7),
             include.lowest = TRUE)
}

F.SEVEN.RAN3 = function(x){
  xnew = cut(x,
             breaks = c(quantile(x, probs = c(0, 0.4, 0.42, 0.44, 0.76, 0.78, 0.80, 1))),
             labels = c(1, 2, 3, 4, 5, 6, 7),
             include.lowest = TRUE)
}

F.SEVEN.RAN4 = function(x){
  xnew = cut(x,
             breaks = c(quantile(x, probs = c(0, 0.35, 0.39, 0.40, 0.75, 0.76, 0.79, 1))),
             labels = c(1, 2, 3, 4, 5, 6, 7),
             include.lowest = TRUE)
}

x1 = F.SEVEN.UNI(v1)
x2 = F.SEVEN.BEL(v2)
x3 = F.SEVEN.RAN1(v3)
x4 = F.SEVEN.RAN2(v4)
x5 = F.SEVEN.RAN3(v5)
x6 = F.SEVEN.RAN4(v6)

myData = data.frame(cbind(x1, x2, x3, x4, x5, x6))


# Check the distribution of each category for each variable.
apply(myData, 2, table)


######################################################
#   Omega Analyses in Different R Packages
######################################################


#~~~~~~~~~~~~~~ R package "MBESS" ~~~~~~~~~~~~~~~~~~#

# Kelley (2017) 

library(MBESS)

# interval.type = "ml" indicates that the method used to find confidence interval is ml.
# Other methods are available depending on the needs of the study.

ci.reliability(data = myData, type = "omega", interval.type = "ml", conf.level = 0.95)

# coefficient alpha
ci.reliability(data = myData, type = "alpha", interval.type = "ml", conf.level = 0.95)


#~~~~~~~~~~~~~~ R package "semTools"  ~~~~~~~~~~~~~~~~~~#

# Pornprasertmanit, Miller, Schoemann, & Rosseel (2013)

# Three omega estimates are available in this package.
# in addition to the conventional coefficient omega reported in the following, 
# an omega estimate that accounted for correlated measurement errors
# and the hierarchical omega can also be calcuated.
# Details can be found in the package manual.

library(lavaan)
model <- 'f =~ x1 + x2 + x3 + x4 + x5 + x6'
fit <- cfa(model, data = myData)

library(semTools)
reliability(fit)[2, 1]

# coefficient alpha
reliability(fit)[1, 1]


#~~~~~~~~~~~~~~ R package "coefficientalpha" ~~~~~~~~~~~~~~~~~~#

# Zhang & Yuan (2015)

# This R package provides robust estimates of alpha and omega 
# as well as the corresponding confidence intervals
# to deal with both outlying observations and missing data.

library(coefficientalpha)

# Varphi refers to the downweighting rate (for the outlying observations);
# by setting varphi at 0, the conventional non-robust omega is calculated.
summary(omega(myData, varphi = 0, se = TRUE))

# coefficient alpha
summary(alpha(myData, varphi = 0, se = TRUE))